/**
 * FB Call Now - Admin JavaScript
 * Version: 4.0.0
 * Theme: SaaS Premium (Buttonizer-inspired)
 */

(function ($) {
    'use strict';

    // Wait for DOM to be ready
    $(document).ready(function () {
        initAdminSettings();
    });

    /**
     * Initialize admin settings functionality
     */
    function initAdminSettings() {
        // Initialize live preview
        initLivePreview();

        // Initialize device toggles
        initDeviceToggles();

        // Initialize button preview for Settings page
        initButtonPreview();

        // Initialize debug log functions
        initDebugLogFunctions();

        // Initialize phone number validation
        initPhoneValidation();

        // Initialize time window validation
        initTimeValidation();

        // Initialize form enhancements
        initFormEnhancements();

        // Initialize settings page navigation
        initSettingsNavigation();
    }

    /**
     * Initialize Device Toggles (Mobile/Desktop)
     */
    function initDeviceToggles() {
        $('.fbcn-device-btn').on('click', function (e) {
            e.preventDefault();

            // UI State
            $('.fbcn-device-btn').removeClass('active');
            $(this).addClass('active');

            // Logic
            var device = $(this).data('device');
            var $stage = $('#fbcn-preview-stage');

            $stage.removeClass('device-mobile device-desktop');
            $stage.addClass('device-' + device);

            // Handle Browser Chrome visibility
            if (device === 'desktop') {
                $('.fbcn-browser-chrome').fadeIn(300);
            } else {
                $('.fbcn-browser-chrome').hide();
            }

            // Re-calculate button position after transition
            setTimeout(updatePreview, 600);
        });
    }

    /**
     * Initialize live preview functionality (Iframe support)
     */
    function initLivePreview() {
        var $iframe = $('#fbcn-preview-iframe');

        if ($iframe.length === 0) {
            return;
        }

        // Wait for iframe to load
        $iframe.on('load', function () {
            var $iframeContents = $iframe.contents();
            var $iframeBody = $iframeContents.find('body');

            // Inject the button styles into the iframe head
            // We use a style tag to ensure font inheritance and scrolled state works properly
            var styleTag = '<style>' +
                'html { margin-top: 0 !important; }' +
                '#wpadminbar { display: none !important; }' +
                '.fbcn-call-button { position: fixed; z-index: 99999; display: inline-flex; align-items: center; justify-content: center; text-decoration: none; border: none; box-shadow: 0 4px 15px rgba(0,0,0,0.15); transition: all 0.5s cubic-bezier(0.4, 0, 0.2, 1); white-space: nowrap; line-height: 1; font-family: inherit; overflow: hidden; }' +
                '.fbcn-call-button:hover { transform: translateY(-2px); box-shadow: 0 8px 25px rgba(0,0,0,0.2); }' +

                /* Scrolled State Logic (Mimics frontend.css) */
                '.fbcn-call-button.scrolled { border-radius: 50% !important; padding: 0 !important; width: 60px !important; height: 60px !important; }' +
                '.fbcn-button-text { display: inline-block; transition: opacity 0.2s ease, transform 0.2s ease; opacity: 1; transform: translateX(0); }' +
                '.fbcn-button-icon { position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%) scale(0); opacity: 0; transition: opacity 0.2s ease, transform 0.2s ease; font-size: 24px; }' +

                '.fbcn-call-button.scrolled .fbcn-button-text { opacity: 0; transform: translateX(-20px); }' +
                '.fbcn-call-button.scrolled .fbcn-button-icon { opacity: 1; transform: translate(-50%, -50%) scale(1); }' +
                '</style>';
            $iframeContents.find('head').append(styleTag);

            // Create the preview button with Icon and Text structure
            var $previewButton = $('<a href="#" id="fbcn-iframe-button" class="fbcn-call-button">' +
                '<span class="fbcn-button-text">' + ($('#button_text').val() || 'Call Now') + '</span>' +
                '<i class="fas fa-phone fbcn-button-icon"></i>' +
                '</a>');

            // Remove any existing buttons (from the active plugin on frontend)
            $iframeContents.find('.fbcn-call-button').remove();

            // Append our preview button
            $iframeBody.append($previewButton);

            // Store reference globally for update function
            window.fbcnPreviewButton = $previewButton;

            // Scroll Logic for Iframe
            // We must attach to the iframe's window object to detect scrolling
            var iframeWindow = $iframe[0].contentWindow;
            $(iframeWindow).on('scroll', function () {
                var scrollTop = $(this).scrollTop();
                if (scrollTop > 50) {
                    $previewButton.addClass('scrolled');
                } else {
                    $previewButton.removeClass('scrolled');
                }
            });

            // Trigger initial update
            updatePreview();
        });

        // Update preview when settings change
        $('#button_text, #button_color, #text_color, [name="fbcn_basic_settings[horizontal_position]"], [name="fbcn_basic_settings[vertical_position]"]').on('input change', function () {
            updatePreview();
        });

        // Also update on color picker change (which might not trigger 'change' directly)
        $('.fbcn-color-picker').wpColorPicker({
            change: function () { setTimeout(updatePreview, 50); },
            clear: function () { setTimeout(updatePreview, 50); }
        });
    }

    /**
     * Update live preview with current form values
     */
    function updatePreview() {
        if (!window.fbcnPreviewButton) {
            return;
        }

        var $preview = window.fbcnPreviewButton;
        var buttonText = $('#button_text').val() || 'Call Now';
        var buttonColor = $('#button_color').val() || '#007cba';
        var textColor = $('#text_color').val() || '#ffffff';

        // Position Logic
        var horizPos = $('[name="fbcn_basic_settings[horizontal_position]"]').val() || 'right';
        var vertPos = parseInt($('[name="fbcn_basic_settings[vertical_position]"]').val() || 10);

        // Calculate bottom percentage based on 1-10 scale
        var topPercent = (vertPos * 9);

        // Refined positioning
        var device = $('#fbcn-preview-stage').hasClass('device-mobile') ? 'mobile' : 'desktop';
        var sideSpacing = device === 'mobile' ? '20px' : '40px';

        // Update the text span, not the main element (to preserve the icon)
        $preview.find('.fbcn-button-text').text(buttonText);

        // Update Styles
        $preview.css({
            'background-color': buttonColor,
            'color': textColor,
            // Use slightly different padding logic since we now have internal spans
            'padding': device === 'mobile' ? '12px 24px' : '14px 32px',
            'font-size': device === 'mobile' ? '16px' : '18px',
            'border-radius': '50px',
            'top': topPercent + '%',
            'font-weight': '600'
        });

        // Position side
        if (horizPos === 'right') {
            $preview.css({ right: sideSpacing, left: 'auto' });
        } else {
            $preview.css({ left: sideSpacing, right: 'auto' });
        }
    }

    /**
     * Initialize phone number validation
     */
    function initPhoneValidation() {
        var $phoneField = $('#phone_number');

        if ($phoneField.length === 0) {
            return;
        }

        $phoneField.on('blur', function () {
            validatePhoneNumber($(this));
        });

        $phoneField.on('input', function () {
            // Remove error styling on input
            $(this).removeClass('fbcn-field-error');
            $('.fbcn-error-icon').remove();
        });
    }

    /**
     * Validate phone number format
     */
    function validatePhoneNumber($field) {
        var phoneValue = $field.val().trim();
        var phoneRegex = /^\+1-\d{3}-\d{3}-\d{4}$/;

        // Remove existing error indicators
        $field.removeClass('fbcn-field-error');
        $field.siblings('.fbcn-error-icon').remove();

        if (phoneValue && !phoneRegex.test(phoneValue)) {
            $field.addClass('fbcn-field-error');
            $field.after('<span class="fbcn-error-icon" title="Invalid phone number format">❗</span>');
            return false;
        }

        return true;
    }

    /**
     * Initialize time window validation
     */
    function initTimeValidation() {
        var $startTime = $('#start_time');
        var $endTime = $('#end_time');
        var $wrapCheckbox = $('[name="fbcn_pro_settings[wrap_to_next_day]"]');

        if ($startTime.length === 0 || $endTime.length === 0) {
            return; // Not on Pro Settings page
        }

        // Validate on change
        $startTime.add($endTime).add($wrapCheckbox).on('change', function () {
            validateTimeWindow();
        });
    }

    /**
     * Validate time window settings
     */
    function validateTimeWindow() {
        var $startTime = $('#start_time');
        var $endTime = $('#end_time');
        var $wrapCheckbox = $('[name="fbcn_pro_settings[wrap_to_next_day]"]');

        var startHour = parseInt($startTime.val().split(':')[0]);
        var endHour = parseInt($endTime.val().split(':')[0]);
        var wrapEnabled = $wrapCheckbox.is(':checked');

        // Remove existing error styling
        $startTime.add($endTime).removeClass('fbcn-field-error');
        $('.fbcn-time-error').remove();

        // Check if start time is later than end time (when not wrapping)
        if (!wrapEnabled && startHour > endHour) {
            $startTime.add($endTime).addClass('fbcn-field-error');
            $endTime.after('<div class="fbcn-time-error notice notice-error inline"><p>Start time cannot be later than end time unless "Wrap to Next Day" is enabled.</p></div>');
            return false;
        }

        return true;
    }

    /**
     * Initialize form enhancements
     */
    function initFormEnhancements() {
        // Add loading state to forms
        $('form').on('submit', function () {
            var $form = $(this);
            var $submitButton = $form.find('input[type="submit"], button[type="submit"]');

            $submitButton.addClass('fbcn-loading');
            $submitButton.prop('disabled', true);

            // Reset after 5 seconds (in case form submission fails)
            setTimeout(function () {
                $submitButton.removeClass('fbcn-loading');
                $submitButton.prop('disabled', false);
            }, 5000);
        });

        // Add confirmation for destructive actions
        $('[name="fbcn_basic_settings[delete_data_on_uninstall]"]').on('change', function () {
            if ($(this).is(':checked')) {
                var confirmed = confirm('Are you sure you want to delete all plugin data when uninstalling? This action cannot be undone.');
                if (!confirmed) {
                    $(this).prop('checked', false);
                }
            }
        });

        // Initialize WordPress color pickers
        initColorPickers();
    }

    /**
     * Initialize WordPress color pickers
     */
    function initColorPickers() {
        if ($.fn.wpColorPicker) {
            $('.fbcn-color-picker').wpColorPicker({
                change: function (event, ui) {
                    // Update live preview when color changes
                    updatePreview();
                },
                clear: function () {
                    // Update live preview when color is cleared
                    updatePreview();
                }
            });
        }
    }

    /**
     * Initialize settings page navigation
     */
    function initSettingsNavigation() {
        // Add active state to current menu item
        var currentPage = getUrlParameter('page');
        $('.wp-submenu a[href*="' + currentPage + '"]').parent().addClass('current');

        // Add smooth scrolling to anchor links
        $('a[href^="#"]').on('click', function (e) {
            e.preventDefault();
            var target = $($(this).attr('href'));
            if (target.length) {
                $('html, body').animate({
                    scrollTop: target.offset().top - 50
                }, 500);
            }
        });
    }

    /**
     * Get URL parameter value
     */
    function getUrlParameter(name) {
        name = name.replace(/[\[]/, '\\[').replace(/[\]]/, '\\]');
        var regex = new RegExp('[\\?&]' + name + '=([^&#]*)');
        var results = regex.exec(location.search);
        return results === null ? '' : decodeURIComponent(results[1].replace(/\+/g, ' '));
    }

    /**
     * Validate entire form before submission
     */
    function validateForm($form) {
        var isValid = true;

        // Validate phone number if present
        var $phoneField = $form.find('#phone_number');
        if ($phoneField.length && !validatePhoneNumber($phoneField)) {
            isValid = false;
        }

        // Validate time window if present
        if ($form.find('#start_time').length && !validateTimeWindow()) {
            isValid = false;
        }

        return isValid;
    }

    /**
     * Add form validation on submit
     */
    $('form').on('submit', function (e) {
        if (!validateForm($(this))) {
            e.preventDefault();

            // Scroll to first error
            var $firstError = $('.fbcn-field-error').first();
            if ($firstError.length) {
                $('html, body').animate({
                    scrollTop: $firstError.offset().top - 100
                }, 300);
                $firstError.focus();
            }

            return false;
        }
    });

    /**
     * Initialize button preview functionality for Settings page
     */
    function initButtonPreview() {
        var $preview = $('#fbcn-preview-button');

        if ($preview.length === 0) {
            return;
        }

        // Force immediate styling on page load
        updatePreview();
    }

    /**
     * Copy log function for Debug Log page
     */
    function initDebugLogFunctions() {
        // Make fbcnCopyLog available globally for the Debug Log page
        window.fbcnCopyLog = function (event) {
            const logContent = document.getElementById('fbcn-log-content');
            if (!logContent) return;

            const textArea = document.createElement('textarea');
            textArea.value = logContent.textContent;
            document.body.appendChild(textArea);
            textArea.select();
            document.execCommand('copy');
            document.body.removeChild(textArea);

            // Show temporary feedback
            const button = event.target;
            const originalText = button.textContent;
            button.textContent = button.getAttribute('data-copied-text') || 'Copied!';
            setTimeout(() => {
                button.textContent = originalText;
            }, 2000);
        };
    }

    /**
     * Public API for external access
     */
    window.FBCallNowAdmin = {
        validateForm: validateForm,
        validatePhoneNumber: validatePhoneNumber,
        validateTimeWindow: validateTimeWindow,
        updatePreview: updatePreview,
        initButtonPreview: initButtonPreview,
        initDebugLogFunctions: initDebugLogFunctions
    };

})(jQuery);